local matrix_utils = {_TYPE='module', _NAME='matrix_utils', _VERSION='0.0.1'}

-- create_transformation_matrix_from_pose gets a pose ([16]float, column-major order))
-- as input and returns a 4x4 matrix
function matrix_utils.create_transformation_matrix_from_pose(pose)
  return matrix { { pose[1], pose[5], pose[9], pose[13] },
                  { pose[2], pose[6], pose[10], pose[14] },
                  { pose[3], pose[7], pose[11], pose[15] },
                  { pose[4], pose[8], pose[12], pose[16] } }
end

-- create_rotation_matrix_from_pose gets a pose ([16]float, column-major order)
-- as input and returns a corresponding 3x3 matrix.
function matrix_utils.create_rotation_matrix_from_pose(pose)
  return matrix { { pose[1], pose[5], pose[9] },
                  { pose[2], pose[6], pose[10] },
                  { pose[3], pose[7], pose[11] } }
end

-- create_pose_from_transformation_matrix gets a 4x4 matrix as input
-- and transforms it into the corresponding pose ([16]float) in
-- column-major order.
function matrix_utils.create_pose_from_transformation_matrix(matrix)
  return { matrix[1][1], matrix[2][1], matrix[3][1], matrix[4][1],
           matrix[1][2], matrix[2][2], matrix[3][2], matrix[4][2],
           matrix[1][3], matrix[2][3], matrix[3][3], matrix[4][3],
           matrix[1][4], matrix[2][4], matrix[3][4], matrix[4][4] }
end

-- Transform given trajectory to given origin. This makes a relative
-- trajectory absolute by "placing" it relative to the passed origin.
-- Parameter trajectory: []{[16]float pose;}
-- Parameter origin: [16]float
-- All in/outputs are column major.
function matrix_utils.make_trajectory_absolute(trajectory, origin)
  local t = {}
  local originTransform = matrix_utils.create_transformation_matrix_from_pose(origin)
  for i=1,#trajectory do
    local point = matrix_utils.create_transformation_matrix_from_pose(trajectory[i].pose)
    t[i] = { pose = matrix_utils.create_pose_from_transformation_matrix(matrix.mul(originTransform, point)) }
  end
  return t
end

-- Transform given trajectory from given origin. This makes a absolute
-- trajectory relative by "subtracting" the passed origin.
-- Parameter trajectory: []{[16]float pose;}
-- Parameter origin: [16]float
-- All in/outputs are column major.
function matrix_utils.make_trajectory_relative(trajectory, origin)
  local t = {}
  local originInv = matrix.invert(matrix_utils.create_transformation_matrix_from_pose(origin))
  for i=1,#trajectory do
    local point = matrix_utils.create_transformation_matrix_from_pose(trajectory[i].pose)
    t[i] = { pose = matrix_utils.create_pose_from_transformation_matrix(matrix.mul(originInv, point)) }
  end
  return t
end

-- Transform given trajectory from from_pose to to_pose.
-- Parameter trajectory: []{[16]float pose;}
-- Parameter from_pose: [16]float
-- Parameter to_pose: [16]float
-- All in/outputs are column major.
function matrix_utils.transform_trajectory(trajectory, from_pose, to_pose)
  local rel = matrix_utils.make_trajectory_relative(trajectory, from_pose)
  return matrix_utils.make_trajectory_absolute(rel, to_pose)
end

return matrix_utils
